(ns ^:no-doc methodical.macros.validate-arities
  "Implementation for `defmethod` arities validation (see [#59](https://github.com/camsaul/methodical/issues/59)).

  The arities sets passed around here are sets of numeric arity counts, or `[:> n]` for arity `n` that also accepts
  varargs, i.e.

  ```clj
  (fn
    ([x])
    ([x y & more]))
  ```

  has the arities `#{1 [:>= 2]}`."
  (:require
   [clojure.data :as data]
   [clojure.set :as set]
   [clojure.spec.alpha :as s]))

(defn- int-between-zero-and-twenty-inclusive? [n]
  (and (integer? n)
       (<= 0 n 20)))

(s/def ::arities-set
  (s/every
   (s/or :int      int-between-zero-and-twenty-inclusive?
         :>=-form  (s/spec (s/cat :>=  (partial = :>=)
                                  :int int-between-zero-and-twenty-inclusive?)))
   :kind set?
   :min-count 1))

(defn- arglist-arities
  "Determine the arity sets for `arglists`. Deals with arglists already conformed using the
  `:clojure.core.specs.alpha/param-list` spec.

  ```clj
  (arglist-arities (s/conform (s/+ :clojure.core.specs.alpha/param-list) '([] [x])))
  => #{0 1}

  (arglist-arities (s/conform (s/+ :clojure.core.specs.alpha/param-list) '([] [x] [x y z & more])))
  => #{0 1 [:>= 3]}
  ```"
  [arglists]
  (into #{}
        (map (fn [{:keys [params var-params]}]
               (if var-params
                 [:>= (count params)]
                 (count params))))
        arglists))

(defn- fn-tail-arities
  "Determine arity sets for `fn-tails`. `fn-tails` should already be conformed using the
  `:methodical.macros/fn-tail` spec or similar:

  ```clj
  (fn-tail-arities (s/conform :methodical.macros/fn-tail '[([x] x) ([x y & more] x)]))
  =>
  #{1 [:>= 2]}
  ```"
  [[arity-type x]]
  (set (case arity-type
         :arity-1 (arglist-arities [(:params x)])
         :arity-n (arglist-arities (map :params x)))))

(defn- expand-arity
  "Given an arity like `1` or `[:> 3]` expand the arity into a flat set of all arities that can be used to invoke a
  function tail with that specific arity, e.g. `[:> 3]` can be invoked with `3`, `4`, `5,` or so on arguments.

  `clojure.lang.IFn/invoke` only supports distinct arities between 0 and 20, inclusive; any more than 20 arguments must
  be invoked with `.applyTo`, so that's all we need to consider here; we'll use the keyword `:more` to represent > 20
  arguments.

  ```clj
  (expand-arity 1)      => #{1}
  (expand-arity [:> 3]) => #{3 4 5 6 7 8 9 ... :more}
  ```"
  [arity]
  (if (integer? arity)
    (sorted-set arity)
    (let [[_ arity] arity]
      (into #{:more} (range arity 21)))))

(defn- expand-arities [arities]
  (into #{} (mapcat expand-arity) arities))

(defn- diff-arities
  "Given a set of `required` arities (e.g., the `:defmethod-arities` in the `defmulti` metadata) and `actual`
  arities (e.g. the function tail arities in a `defmethod` form, as generated by [[fn-tail-arities]]), return a map with
  any `:required` arities that are missing and any `:disallowed` arities that are present. Returns `nil` if there are no
  missing required arities or disallowed arities present."
  [required actual]
  (let [[missing disallowed] (data/diff (expand-arities required) (expand-arities actual))]
    (not-empty
     (into {} (for [[k orig expanded] [[:required required missing]
                                       [:disallowed actual disallowed]]
                    :when             (seq expanded)]
                [k (set (for [arity orig
                              :when (not-empty (set/intersection (expand-arity arity) expanded))]
                          arity))])))))

(defn allowed-arities-fn-tail-spec
  "Create a spec for a function tail to make sure it has all of the `required-arities`, and all of its arities are
  allowed."
  [required-arities]
  (if (empty? required-arities)
    identity
    (s/and (s/conformer (fn [fn-tail]
                    (let [arities (fn-tail-arities fn-tail)
                          diff    (diff-arities required-arities arities)]
                      (when (seq diff)
                        {:arities diff}))))
           empty?)))
