# H2 Database Dialect - Key Differences

H2 is a lightweight Java SQL database, commonly used for testing.

## Identifier Quoting

Double quotes for identifiers. **Uppercases unquoted identifiers** by default.

```sql
SELECT "CamelCase", "select" FROM "My Table"
SELECT myColumn FROM t    -- Actually selects "MYCOLUMN"
```

**Schema-qualified names:** Schema and table are SEPARATE identifiers. Quote each part individually if needed:

```sql
-- CORRECT: each identifier quoted separately
SELECT * FROM "PUBLIC"."PRODUCTS"
SELECT * FROM PUBLIC.PRODUCTS         -- unquoted is uppercased

-- WRONG: quoting the whole path as one identifier
SELECT * FROM "PUBLIC.PRODUCTS"       -- Looks for table literally named "PUBLIC.PRODUCTS"
```

## Compatibility Modes

H2 can emulate other databases via connection URL:

```
jdbc:h2:mem:test;MODE=PostgreSQL
jdbc:h2:mem:test;MODE=MySQL
```

Available: REGULAR, PostgreSQL, MySQL, Oracle, MSSQLServer, DB2, Derby, HSQLDB, MariaDB.

## SYSTEM_RANGE (Generate Series)

```sql
SELECT x FROM SYSTEM_RANGE(1, 10)                 -- 1 to 10

-- Date range
SELECT DATEADD('DAY', x, DATE '2024-01-01') AS dt
FROM SYSTEM_RANGE(0, 364)
```

## CASEWHEN (H2-Specific Ternary)

```sql
SELECT CASEWHEN(status = 'active', 'Yes', 'No')
SELECT CASE WHEN status = 'active' THEN 'Yes' ELSE 'No' END   -- Standard also works
```

## Arrays (1-Indexed)

```sql
SELECT my_array[1] AS first_element               -- 1-indexed!
SELECT CARDINALITY(arr)                           -- Length
SELECT ARRAY_CONTAINS(arr, value)

-- Flatten
SELECT * FROM UNNEST(ARRAY[1, 2, 3])
```

## Date Functions

```sql
SELECT DATE_TRUNC('MONTH', order_date)
SELECT DATEADD('DAY', 7, order_date)
SELECT DATEDIFF('DAY', start_date, end_date)

-- Java format patterns
SELECT FORMATDATETIME(order_date, 'yyyy-MM-dd')
SELECT PARSEDATETIME('2024-01-15', 'yyyy-MM-dd')
```

## String Aggregation

```sql
SELECT LISTAGG(name, ', ') FROM t GROUP BY category
SELECT GROUP_CONCAT(name SEPARATOR ', ') FROM t GROUP BY category   -- MySQL mode
```

## MERGE for Upsert

```sql
MERGE INTO target_table t
USING source_table s ON t.id = s.id
WHEN MATCHED THEN UPDATE SET t.name = s.name
WHEN NOT MATCHED THEN INSERT (id, name) VALUES (s.id, s.name)
```

## VALUES (Inline Table)

```sql
SELECT * FROM (VALUES (1, 'a'), (2, 'b'), (3, 'c')) AS t(id, name)
```

## JSON (H2 2.x)

```sql
SELECT JSON_VALUE(json_col, '$.name')
SELECT json_col.name                              -- Dot notation
```

## No QUALIFY - Use Subquery

```sql
SELECT * FROM (
  SELECT *, ROW_NUMBER() OVER (PARTITION BY category ORDER BY amount DESC) AS rn
  FROM products
) WHERE rn = 1
```
